#include "monitorwidget.h"

/*!
 * This constructor sets up tabs in the central primary window and adds a tab to it.It also initializes a new instance of TableModel type.
 */
MonitorWidget::MonitorWidget(QWidget *parent)
    : QTabWidget(parent)
{
    //Adding System information tab
    table = new TableModel(this);
    sysInfoTab = new SysInfoTab(this);

    connect(sysInfoTab, SIGNAL(sendDetails(QString, QString, QString, QString)),
        this, SLOT(addEntry(QString, QString, QString, QString)));

    addTab(sysInfoTab, "System Info");

    //Adding Process tab
    setupTabs();

    //Adding CPU Utilization tab
    image = new QImage();
    btn = new QToolButton;
    btn->setIconSize(resultSize);

    p.get_load();   //change the image according to current load
    loadImage("./graph.png", image, btn);
    addTab(btn, "CPU Utilization");

    //Timer for continiously loading the images for the graph
    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(refreshImage()));
    timer->start(1000);
}

void MonitorWidget::addEntry(QString name, QString pid, QString cpu, QString owner)
{
    QList< Process >list = table->getList();
    Process proc(name.toStdString(), pid.toStdString(), cpu.toStdString(), owner.toStdString());

    if (!list.contains(proc)) {
        table->insertRows(0, 1, QModelIndex());

        QModelIndex index = table->index(0, 0, QModelIndex());
        table->setData(index, name, Qt::EditRole);
        index = table->index(0, 1, QModelIndex());
        table->setData(index, pid, Qt::EditRole);
        removeTab(indexOf(sysInfoTab));
    } else {
        QMessageBox::information(this, tr("Duplicate Name"),
            tr("The name \"%1\" already exists.").arg(name));
    }
}

/*!
 * This function is called when the timer signal is raised to update the graph
 * image.
  */
void MonitorWidget::refreshImage()
{
    p.get_load();
    loadImage("./graph.png", image, btn);
}

/*!
 * This function deletes all rows in the table
 */
void MonitorWidget::removeEntry()
{
    QTableView *temp = static_cast<QTableView*>(currentWidget());
    QSortFilterProxyModel *proxy = static_cast<QSortFilterProxyModel*>(temp->model());
    QItemSelectionModel *selectionModel = temp->selectionModel();

    QModelIndexList indexes = selectionModel->selectedRows();
    QModelIndex index;

    foreach (index, indexes) {
        int row = proxy->mapToSource(index).row();
        table->removeRows(row, 1, QModelIndex());
    }

    if (table->rowCount(QModelIndex()) == 0) {
        insertTab(0, sysInfoTab, "System Info");
    }
}

/*!
 * This function sets all the right properties of the tabs in the primary window. It also raises a call to the rowSelected function in the
 *  event of a row being double clicked.
 */
void MonitorWidget::setupTabs()
{

        QString str = "Processes";

        proxyModel = new QSortFilterProxyModel(this);
        proxyModel->setSourceModel(table);
        proxyModel->setDynamicSortFilter(true);

        QTableView *tableView = new QTableView;
        tableView->setModel(proxyModel);
        tableView->setSortingEnabled(true);
        tableView->setSelectionBehavior(QAbstractItemView::SelectRows);
        tableView->horizontalHeader()->setStretchLastSection(true);
        tableView->verticalHeader()->hide();
        tableView->setEditTriggers(QAbstractItemView::NoEditTriggers);
        tableView->setSelectionMode(QAbstractItemView::SingleSelection);

        QString newStr = QString("^[%1].*").arg(str);

        proxyModel->setFilterRegExp(QRegExp(newStr, Qt::CaseInsensitive));
        proxyModel->setFilterKeyColumn(0);
        proxyModel->sort(0, Qt::AscendingOrder);

        connect(tableView, SIGNAL(doubleClicked(QModelIndex)),table, SLOT(rowSelected(QModelIndex)));
        addTab(tableView, str);

}

/*!
 * Function to load graph image into CPU utilization
 */
void MonitorWidget::loadImage(const QString &fileName, QImage *image,
                              QToolButton *button)
{

    image->load(fileName);
    QImage fixedImage(resultSize, QImage::Format_ARGB32_Premultiplied);
    QPainter painter(&fixedImage);
    painter.setCompositionMode(QPainter::CompositionMode_Source);
    painter.fillRect(fixedImage.rect(), Qt::transparent);
    painter.setCompositionMode(QPainter::CompositionMode_SourceOver);
    QPoint q = QPoint((resultSize.width() - image->width()) / 2, (resultSize.height() - image->height()) / 2);
    painter.drawImage(q, *image);
    painter.end();
    *image = fixedImage;
    button->setIcon(QPixmap::fromImage(*image));
}
