#include "tablemodel.h"

/*!
 * Slot being called when timer signal is raised.
 */
void TableModel::refreshTable()
{
    QModelIndex index;
    removeRows(0, listOfProcesses.size(), index);
    refillRows(index);
}

/*!
 * Constructor. Initializes a ProcessExtractor type object, populates a list
 * by function calls to select rows from database. Raises signal every 3 seconds
 * to call refreshTable() so that list of running processing is refrehed in the
 * database and hence the table.
 */
TableModel::TableModel(QObject *parent) : QAbstractTableModel(parent)
{
    ProcessExtractor p;
    p.addToDatabase();
    listOfProcesses = p.getList();
    timer = new QTimer(this);
    QObject::connect(timer, SIGNAL(timeout()), this, SLOT(refreshTable()));
    timer->start(3000);
}

/*!
 * Contructor to assign data member listOfProcesses with @param pairs
 */
TableModel::TableModel(QList< Process > pairs, QObject *parent)
        : QAbstractTableModel(parent)
{
    listOfProcesses=pairs;
}

TableModel::~TableModel()
{
    delete timer;
}

/*!
 * Returns the size of listOfProcesses
 */
int TableModel::rowCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent);
    return listOfProcesses.size();
}

/*!
 * Returns the number of columns in database and table
 */
int TableModel::columnCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent);
    return 4;
}

/*!
 * Depending on the @param index passed, function returns appropriate value in
 * corresponding row-column pair
 */
QVariant TableModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid())
        return QVariant();
    
    if (index.row() >= listOfProcesses.size() || index.row() < 0)
        return QVariant();
    
    if (role == Qt::DisplayRole) {
        Process tempProc = listOfProcesses.at(index.row());
        
        if (index.column() == 0)
            return QString(tempProc.getName().c_str());
        else if (index.column() == 1)
            return QString(tempProc.getPid().c_str());
        else if (index.column() == 2)
            return QString(tempProc.getCpu().c_str());
        else if (index.column() == 3)
            return QString(tempProc.getOwner().c_str());
    }
    return QVariant();
}

/*!
 * Function to assign header information on the table
 */
QVariant TableModel::headerData(int section, Qt::Orientation orientation, int role) const
{
    if (role != Qt::DisplayRole)
        return QVariant();
    
    if (orientation == Qt::Horizontal) {
        switch (section) {
        case 0:
            return tr("Owner");

        case 1:
            return tr("PID");

        case 2:
            return tr("CPU %");

        case 3:
            return tr("Process Name");

        default:
            return QVariant();
        }
    }
    return QVariant();
}

/*!
 * Function to insert empty rows into the table
 */
bool TableModel::insertRows(int position, int rows, const QModelIndex &index)
{
    Q_UNUSED(index);
    beginInsertRows(QModelIndex(), position, position+rows-1);
    
    for (int row=0; row < rows; row++) {
        Process tempProc(" "," "," "," ");
        listOfProcesses.insert(position, tempProc);
    }

    endInsertRows();
    return true;
}

/*!
 * Function called when timer signal raised. Refrehes database and table rows
 * with afresh with latest process information
 */
bool TableModel::refillRows(const QModelIndex &index)
{

    Q_UNUSED(index);
    int position=0;
    ProcessExtractor p;
    p.addToDatabase();
    listOfProcesses.clear();
    QList<Process> testList = p.getList();
    int rows=testList.size();
    beginInsertRows(QModelIndex(), position, position+rows-1);
    listOfProcesses=testList;
    endInsertRows();
    return true;
}

/*!
 * Function to remove @param rows number of rows  from given @param position
 */
bool TableModel::removeRows(int position, int rows, const QModelIndex &index)
{
    Q_UNUSED(index);    
    beginRemoveRows(QModelIndex(), position, position+rows-1);
    
    for (int row=0; row < rows; ++row) {
        listOfProcesses.removeAt(position);
    }

    endRemoveRows();
    return true;
}

/*!
 * Sets @class Process information to current instance from retrieved values of
 * @param value
 */
bool TableModel::setData(const QModelIndex &index, const QVariant &value, int role)
{
    if (index.isValid() && role == Qt::EditRole) {
        int row = index.row();

        Process p = listOfProcesses.value(row);

        if (index.column() == 0)
            p.setName(value.toString().toStdString());
        else if (index.column() == 1)
            p.setPid(value.toString().toStdString());
        else if (index.column() == 2)
            p.setCpu(value.toString().toStdString());
        else if (index.column() == 3)
            p.setOwner(value.toString().toStdString());
        else
            return false;

        listOfProcesses.replace(row, p);
        emit(dataChanged(index, index));

        return true;
    }
    return false;
}

/*!
 * Function to set flag values
 */
Qt::ItemFlags TableModel::flags(const QModelIndex &index) const
{
    if (!index.isValid())
        return Qt::ItemIsEnabled;
    
    return QAbstractTableModel::flags(index) | Qt::ItemIsEditable;
}

QList< Process > TableModel::getList()
{
    return listOfProcesses;
}

/*!
 * This is a slot raised by signal of row being double clicked in a table.
 * It initiates the pop up window displaying all files used by the selected
 * process.
 */
void TableModel::rowSelected(QModelIndex index)
{
    if(index.row()==-1)
        return;
    QModelIndex index1=QAbstractItemModel::createIndex(index.row(),1);
    lsof_pid = index1.data().toString().toStdString();
    fileList.lsofPopulate(lsof_pid);
    fileList.updatePreview();
}
