/**This program acts like a server in the LAN Messenger.
  *It listens to the user specified port and accepts all client connections and 
  *then it would list all the messages being sent by the clients, log them into a database and list of all
  *connected clients.
  *@file server.c
  *@author Ratna Kumar K.V.R
  */

#include "server.h"

/**This function is used for closing a file.
  *It returns an error message if it cannot close the file/socket.
  *@param fd file descriptor of the file/socket to be closed.
  */
 
void Close(int fd)
{
    int return_value;
    return_value=close(fd);
    if(return_value<0)
    {
	fprintf(stderr, "Cannot close\n");
    }
}

/**This function is called when we press Ctrl+C key or on any error condition.
  *It does proper closing of all file descriptors and database connections.
   @param  signal (SIGINT)
  */
void close_properly(int signal)
{
    int i;
    //shutting down all the client connections..
    for(i = 0; i <= maxi; i++)
    {
	if(client[i].fd < 0)
		continue;
	shutdown(client[i].fd, 2);
    }

   //closing the mysql database connection
    mysql_close(connection1);
   //closing the listen file descriptor
    Close(listen_file_descriptor);
    exit(0);
}

/**This function is used for writing into the file/socket.
  *It prints error message and returns the status if it fails to write.
  *@param fd file descriptor of the file into which we write.
  *@param buf pointer to the buffer which stores the data to be written.
  *@param count number of bytes to be written.
  *@return number of bytes written ( < 0 indicates failure) 
  */
ssize_t Write(int fd, const void *buf, size_t count)
{
    ssize_t return_value;
    return_value=write(fd,buf,count);
    if(return_value < 0)
    {
	fprintf(stderr, "Write failed\n");
    }
    return return_value;
}

/**This function creates an unbound socket in a communications domain, and 
  *return a file descriptor that can be used in later function calls that 
  *operate on sockets.
  *@param domain Specifies the communications domain in which a socket is to be created.
  *@param type Specifies the type of socket to be created.
  *@param protocol Specifies a particular protocol to be used with the socket. 
  *@return non-negative integer, the socket file descriptor. Otherwise, a value of -1 shall be returned and errno set to indicate the error.
  */
int Socket(int domain, int type, int protocol)
{
    int return_value;
    return_value = socket(domain, type, protocol);
    if(return_value < 0)
    {
	strcpy(err_msg, "Cannot open socket.\n");
	success = 0;
    }
    return return_value;
}

/**This function shall assign a local socket address address to a socket identified 
  *by descriptor socket that has no local socket address assigned.
  *@param sockfd Specifies the file descriptor of the socket to be bound.
  *@param my_addr Points to a sockaddr structure containing the address to be bound to the socket.
  *@param addrlen Specifies the length of the sockaddr structure pointed to by the address argument. 
  */
void Bind(int  sockfd,  const  struct  sockaddr *my_addr, socklen_t addrlen)
{
    int return_value;
    return_value= bind(sockfd,my_addr,addrlen);
    if(return_value < 0)
    {
	strcpy(err_msg, "Cannot bind");
	success = 0;
    }
}

/**This function shall mark a connection-mode socket, specified by the socket argument, as accepting connections
  *@param sockfd file descriptor on which it has to listen.
   @param backlog used to limit the number of outstanding connections in the socket's listen queue.
  */
void Listen(int sockfd, int backlog)
{
    int return_value;
    return_value = listen(sockfd, backlog);
    if(return_value < 0)
    {
	strcpy(err_msg, "Cannot listen");
	success = 0;
    }
}

/**This function hall extract the first connection on the queue of pending connections, 
  *create a new socket with the same socket type protocol and address family as the specified socket, 
  *and allocate a new file descriptor for that socket.
  *@param sockfd Specifies a socket that was created with socket(), has been bound to an address with bind(), and has issued a successful call to listen().
  *@param addr Either a null pointer, or a pointer to a sockaddr structure where the address of the connecting socket shall be returned.
  *@param addrlen Points to a socklen_t structure which specifies the length of the supplied sockaddr structure.
  *@return the non-negative file descriptor of the accepted socket. Otherwise, -1 shall be returned and errno set to indicate the error.
  */
int Accept(int sockfd, struct sockaddr *addr, socklen_t *addrlen)
{
    int file_descriptor;
    file_descriptor = accept(sockfd, addr, addrlen);
    if(file_descriptor < 0)
    {
	fprintf(stderr, "accept() failed.");
	exit(1);
    }
    return file_descriptor;
}


/**This function reads the data from file/socket into a buffer.
  *@param fd file descriptor used for reading data
  *@param buf buffer into which the data read from file is stored
  *@param count Maximum number of bytes to be read
  *@return number of bytes read ( < 0 indicates error)
  */
ssize_t Read(int fd, void *buf, size_t count)
{
    ssize_t characters_read;
    characters_read = read(fd, buf, count);
    if(characters_read < 0)
    {
	fprintf(stderr, "Can't read\n");
    }
    return characters_read;
}

/**This function checks if the client who is attempting to connect with a nick name
  *already exists or not.
  *@param nick nickname of client attempting to connect
  *@return 1 if nick already exists, 0 if not exists.
  */
int isNickExists(char * nick)
{
	int i;
	//checking with all connected clients if the nick name already exists.
	for(i = 0; i <= maxi; i++)
	{
		if(client[i].fd < 0)
		{
			continue;
		}
		if(strcasecmp(client[i].nick, nick) == 0)
		{
			return 1;		
		}
	}

	return 0;
}

/**This function inserts a row into a mysql database. The row contains information like 
  *ID, timestamp, message of client, nickname of client, IP address of client. basically logging the chat into database.
  *@param query insert query.
  */
void insert_row(char * query)
{
	int result;
	
	result = mysql_query (connection1, query);
	if (result!=0)
	{
	    fprintf (stderr, "Could not insert data as %d: %s\n",
		     mysql_errno (connection1),
		     mysql_error (connection1));
	    close_properly(1);
	}
}

/**This function sends the message send by one client to all other connected clients.
  *@param data message to be sent.
  */
void write_all(char *data)
{
	int i;
	const time_t timer = time(NULL);// for getting time
    	char tmp[BUFFER]; // temporary variable used for message formatting
	char tm[BUFFER];// temporary variable used for timestamp
	char query[BUFFER];// used for generating a SQL query for inserting a row into database
	struct transmit_unit trs_blk; // transmitting unit block.

	strcpy(tm, ctime(&timer));
	tm[strlen(tm) - 1] = 0;

	//copying client information into transmission block
	strcpy(trs_blk.ip, ip);
	strcpy(trs_blk.nick, nick);
	strcpy(trs_blk.time, tm);
	sprintf(tmp, "%s", data);
	strcpy(trs_blk.msg, tmp);

	
	sprintf(tmp, "[%s] <%s>:%s\n", tm, nick, data);
	gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, tmp, -1, "blue_fg", "lmarg",  NULL); // displayin message in ChatWindow

	//generating insert query 
	sprintf(query, "insert into cs3002_table (timestamp, message, client_ip, client_nick) VALUES ('%s', '%s', '%s', '%s')", tm, data, ip, nick); 
	insert_row(query); // inserting row into database

	//sending message to all connected clients..
	for(i = 0; i <=maxi; i++)
	{
		if(client[i].fd < 0)
			continue;
		Write(client[i].fd, &trs_blk, sizeof(struct transmit_unit));
	}
}

/**This function adds the widget to the Box with corresponding label passed
  *@param box  Box into which widget has to be packed
  *@param caption label to be given.
  *@param widget widget to be packed. 
  */
void add_widget_with_label(GtkContainer *box, gchar *caption, GtkWidget *widget)
{
    GtkWidget *label = gtk_label_new(caption);
    GtkWidget *hbox = gtk_hbox_new(TRUE, 4);

    gtk_container_add(GTK_CONTAINER (hbox), label);
    gtk_container_add(GTK_CONTAINER (hbox), widget);
    gtk_container_add(box, hbox);
}

/**This function calls when we close the window.. which quits the window and exits the program.
  *It shutdowns and closes all the connections.
  *@param widget1 Widget to be closed
  *@param data pointer 
  */
void end_program(GtkWidget *widget1, gpointer data)
{
	int i;
	//shutting down all the connections..
	for(i = 0; i <= maxi; i++)
	{
		if(client[i].fd < 0)
			continue;

		shutdown(client[i].fd, 2);
	}

	//closing mysql database connection
	mysql_close(connection1);
	//closing listen socket
	Close(listen_file_descriptor);	
	//qutting the GTK window..
    	gtk_main_quit();
}

/**This function calls when we close the window.. which quits the window and exits the program.
  *so it shutdowns and closes all the connections.
  *@param widget1 Widget to be closed
  *@param data pointer 
  */
void onExit(GtkWidget *widget1, gpointer data)
{
	exit(1);
}


/**This function connects to the database and creates a table (if not present) for logging the chats of clients.
  */
void mysql_connect_create_table()
{
	int result; // for getting the result status
	connection1 = mysql_init (NULL); // getting the mysql database handler
    	if(mysql_real_connect(connection1, "localhost", "cs3002_user", "cs3002_password", "cs3002_db", 0, NULL, 0))//connecting to database
    	{
		printf ("Database Connection successful\n");		
		result = mysql_query (connection1, "create table cs3002_table (id integer NOT NULL AUTO_INCREMENT PRIMARY KEY, timestamp varchar(4096), message varchar(4096), client_ip varchar(4096), client_nick varchar(4096))");
    	}
        else
    	{
		fprintf (stderr, "Connection failed\n");
		if (mysql_errno (connection1))
		{
		    fprintf (stderr, "Connection error %d: %s\n",
		     mysql_errno (connection1),
		     mysql_error (connection1));
		}
		exit(1);
    	}
}

/**This function stars the server with the given port for listening.It is called
  *when we press "Connect" button Widget. It gives appropriate message if it cannot 
  *listen to the port.
  *@param widget1 Connect button
  *@param data pointer
  */
void connect_server(GtkWidget *widget1, gpointer data)
{
	struct sockaddr_in server_address;
	GtkWidget *dialog;
	success = 1; // used for checking whether server is able to listen or not.
	printf("Server Starting..\n");
	listen_file_descriptor = Socket(AF_INET, SOCK_STREAM, 0);//creating a socket.
	if(success == 1)
	{
		bzero(&server_address, sizeof(server_address));// initlizing the server_address structure
	        server_address.sin_family = AF_INET; // assing the family.
        	server_address.sin_addr.s_addr  = htonl(INADDR_ANY);//converting the host address to network format
        	server_address.sin_port = htons(atoi(gtk_entry_get_text(GTK_ENTRY(entry_port))));// assigning the port number given by user
	        Bind(listen_file_descriptor, (struct sockaddr *) &server_address, sizeof(server_address));//binding..
	}
	
	if(success == 1)
	{
	        Listen(listen_file_descriptor, 10); // Listening on the given port.
	}
	if(success == 1)
	{
		gtk_main_quit(); // if it is able to listen then quit the authentication window and open chat room..
	}
	else
	{
		dialog = gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_INFO, GTK_BUTTONS_OK,"%s", err_msg);//open dialog box with apprpriate message..
		gtk_dialog_run (GTK_DIALOG (dialog));
        	gtk_widget_destroy (dialog);	
	}		
}

/**This function starts the server. It accepts the client connections and sends the client messages to all other
  *connected clients, logs the chat into database and lists the connected clients.
  *@param arg arguments to be sent
  */
void * start_server(void * arg)
{
	int connection_file_descriptor; //temporary variable used for storing client connection
	int socket_file_descriptor; //temporary variable used for storing client connection
	int nready; // number of ready descriptors which is used for processing
	char input_data[BUFFER]; // used to getting the storing data
	char temp_buffer[BUFFER]; // temporary variable used for storing data
	int i, k, maxfd, n; // temporary variables used..
	gboolean     valid; // used for traversing the connected client list..
	gchar *nick_name; // used for getting nick name
	
	socklen_t clilen;//used for storing client length

	struct sockaddr_in client_address; //used for storing client address information
	struct sigaction act1; // used for registering signals.
	struct client_data end_client; // temporary variable used for indicating end of connected client list.

        fd_set read_set, all_set; // used in select for storing read set of file descriptors.

	gtk_text_buffer_insert_with_tags_by_name(buffer, &iter, "Server started...\n", -1, "blue_fg", "lmarg",  NULL);

	//close connections when Ctrl+C is pressed
	act1.sa_handler = close_properly;
	sigemptyset(&act1.sa_mask);
	act1.sa_flags=0;
	sigaction(SIGINT, &act1, 0);

	//initilizing..
	maxfd = listen_file_descriptor;
	maxi = -1;

	for(i = 0; i < FD_SETSIZE; i++)
	{
		client[i].fd = -1;
		strcpy(client[i].ip, "");
		strcpy(client[i].nick,"");		
	}
	
	FD_ZERO(&all_set);
	FD_SET(listen_file_descriptor, &all_set);
	
	while(1)
	{		
		read_set = all_set;
		nready = select(maxfd+1, &read_set, NULL, NULL, NULL);

		//new connection
		if(FD_ISSET(listen_file_descriptor, &read_set))
		{
			clilen = sizeof(client_address);
			
			connection_file_descriptor = Accept(listen_file_descriptor, (struct sockaddr *) &client_address, &clilen);
			
			if((n = Read(connection_file_descriptor, input_data, BUFFER)) == 0)
			{
				printf("Failed init\n");
				shutdown(connection_file_descriptor, 2);
			}
			else
			{
				input_data[n] = '\0';
				if(isNickExists(input_data)) // if nick alrady exists.. shut down the client connection..
				{
					printf("Failed in nick\n");
					shutdown(connection_file_descriptor, 2);
				}
				else
				{					
					for (i = 0; i < FD_SETSIZE; i++)
					{
					    if (client[i].fd < 0) //find free space..
					    {
						 //store client information.
					         client[i].fd = connection_file_descriptor; /* save descriptor */
						 strcpy(client[i].ip, inet_ntoa(client_address.sin_addr));
						 strcpy(client[i].nick, input_data);

						//adding the client to the list..
						 gtk_list_store_append(list_store, &t_iter);
						 gtk_list_store_set (list_store, &t_iter, 0, client[i].nick, 1, client[i].ip, -1);
						  
					         break;
    					    }
					}
		
					if(i == FD_SETSIZE) // if client LIMIT exceeded then shut down
					{
						shutdown(connection_file_descriptor, 2);
					}
					else
					{
						//adding new descriptor to the set..
						FD_SET(connection_file_descriptor, &all_set);
		
						//setting 1st arg in select..
						if(connection_file_descriptor > maxfd)
						{
							maxfd = connection_file_descriptor;
						}
		
						//setting max index in the client array
						if(i > maxi)
						{
							maxi = i;
						}
				
						//indicating end of connected client list sent to newly connected client..
						end_client.fd = -1;
						strcpy(end_client.ip, "");
						strcpy(end_client.nick, "");

						//send the list of connected clients to the newly connected client.
						for(k = 0; k <= maxi; k++)
						{
							 if(client[k].fd < 0)
								continue;

							  Write(connection_file_descriptor, &client[k], sizeof(struct client_data));
						}
						//indicating end of list..
						Write(connection_file_descriptor, &end_client, sizeof(struct client_data));

						for(k = 0; k <= 50; k++); // wait 

						sprintf(temp_buffer, "connected..");
						strcpy(nick, client[i].nick);
						strcpy(ip, client[i].ip);
						write_all(temp_buffer);// send the message to all the clients about the new client connection
					}
				}
			}

			//if no more readable descriptors..
			if(--nready <= 0)
			{
				continue;
			}
               }

			//checking all the clients for data..
	       for(i = 0; i <= maxi; i++)
	       {
			if((socket_file_descriptor = client[i].fd) < 0)
			{
				continue;
			}
			
			if(FD_ISSET(socket_file_descriptor, &read_set))
			{
				    if ((n = Read(socket_file_descriptor, input_data, BUFFER)) <= 0) 
				    {
			            	/* connection closed by client */
			            	Close(socket_file_descriptor);
		           	    	FD_CLR(socket_file_descriptor, &all_set);

					sprintf(temp_buffer, "disconnected..");
					strcpy(nick, client[i].nick);
					strcpy(ip, client[i].ip);
			
					//clearing the corresponding data structures...
			            	client[i].fd = -1;
					strcpy(client[i].nick ,"");
					strcpy(client[i].ip,"");
				 	//need to write to all clients about the client disconnection...
					write_all(temp_buffer);
	
					//removing the client from the list..
					valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(list_store), &t_iter);
					while(valid)
					{
						gtk_tree_model_get(GTK_TREE_MODEL(list_store), &t_iter, 0, &nick_name, -1);
						if(strcmp(nick_name, nick) == 0)
						{
							gtk_list_store_remove(list_store, &t_iter);
							g_free(nick_name);
							break;
						}
						valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(list_store), &t_iter);
						g_free(nick_name);
					}	

				    }
				    else
				    {
					//Message sent by client..
					input_data[n] = '\0';
					sprintf(temp_buffer, "%s", input_data);
					strcpy(nick, client[i].nick);
					strcpy(ip, client[i].ip);
				 	//need to send the message to all connected clients ...
					write_all(temp_buffer);
		                    }
		
				    //if no more readable descriptors..
	  		            if (--nready <= 0)
				    {
					break;
				    }
			}
		}
	}	
	
	
}

/** This is where program execution starts. All arguments are passed
 * to gtk for processing, which removes any arguments relevant to it.
 * All other arguments are ignored. This function first gets a window for 
 * user to enter listening port. If it can listen succesfully then it displays the 
 * chat room, else it displays the dialog box with approriate message.
 * @param argc Number of arguments passed.
 * @param argv NULL terminated array of command line arguments.
 * @return EXIT_SUCCESS on success, EXIT_FAILURE on failure
 */
int main(int argc, char * argv[])
{
	GtkWidget *window;	//the main window
	GtkWidget *window1;	//the main window
	GtkWidget *mainbox;	//Box 
	GtkWidget *view1;	//view for text buffer (Chat Window)
	GtkWidget *view2;	//view for List Window
	GtkWidget *scrolled_window1; // scrolled window for chat
	GtkWidget *scrolled_window2; // scrolled window for list
	GtkWidget *button_connect; // connect button
    	GtkWidget *vbox; // Box
	GtkCellRenderer     *renderer; // used in Lists.

	pthread_t child1; // used for threading
	int return_value; // getting the status

	gtk_init(&argc, &argv);


	//creating window for entering listening port..
	window1 = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    	gtk_window_set_default_size(GTK_WINDOW(window1), 300, 200);
	g_signal_connect(GTK_OBJECT(window1), "delete_event", GTK_SIGNAL_FUNC(onExit), NULL);

	vbox = gtk_vbox_new(TRUE, 5);
	entry_port = gtk_entry_new();
	button_connect = gtk_button_new_with_label("Connect");

	add_widget_with_label(GTK_CONTAINER(vbox), "Enter the Listening Port Number : ", entry_port);
        gtk_box_pack_start(GTK_BOX(vbox), button_connect, TRUE, FALSE, 5);
	gtk_container_add(GTK_CONTAINER(window1), vbox);
        g_signal_connect(GTK_OBJECT(button_connect), "clicked", GTK_SIGNAL_FUNC(connect_server), NULL);

	gtk_widget_show_all(window1);
        gtk_main();

	//If server can listen the port then destroy this window and display the Chat Room window.
	gtk_widget_destroy (window1);


	//creating window
    	window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), "LAN-Messenger Server");
	gtk_window_set_default_size(GTK_WINDOW(window),1040,600);
	g_signal_connect(G_OBJECT(window),"delete_event",G_CALLBACK(end_program),NULL);

	//creating horizantal box main box.. 
        mainbox = gtk_hbox_new (TRUE, 2);

	/* Create the GtkText View widget */
	view1 = gtk_text_view_new();
	gtk_text_view_set_editable(GTK_TEXT_VIEW(view1), FALSE);
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(view1));
	gtk_text_buffer_get_iter_at_offset(buffer, &iter, 0);	
	gtk_text_buffer_create_tag(buffer, "blue_fg", "foreground", "blue", NULL); 
	gtk_text_buffer_create_tag(buffer, "lmarg", "left_margin", 5, NULL);

	/* This is the scrolled window to put the text buffer widget inside */
	scrolled_window1=gtk_scrolled_window_new(NULL, NULL);
        //gtk_widget_set_usize(scrolled_window1, 250, 150);
	gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW(scrolled_window1), view1);


	/*creating a tree view of list model for showing list of all connected clients*/
	view2 = gtk_tree_view_new ();
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view2), -1, "Nick", renderer, "text", 0, NULL);
	
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view2), -1, "Ip Address", renderer, "text", 1, NULL);

	list_store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
	gtk_tree_view_set_model (GTK_TREE_VIEW (view2), GTK_TREE_MODEL(list_store));
	

	/* This is the scrolled window to put the List widget inside */
	scrolled_window2=gtk_scrolled_window_new(NULL, NULL);
        //gtk_widget_set_usize(scrolled_window2, 250, 150);
	gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW(scrolled_window2), view2);

	gtk_box_pack_start (GTK_BOX (mainbox), scrolled_window1, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (mainbox), scrolled_window2, TRUE, TRUE, 0);	
	gtk_container_add (GTK_CONTAINER (window), mainbox);
	gtk_widget_show_all (window);

	mysql_connect_create_table();// connect to the mysql database and create a table (if not exist)	
		
	//Create thread for starting the server..
	return_value = 	pthread_create(&child1, NULL, start_server, NULL);
	if(return_value !=0)
	{
	    perror("Thread 1  creation failed");
	    exit(EXIT_FAILURE);
	}
	
	gtk_main();
	return 0;
}
